<?php

declare(strict_types=1);

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Http\Requests\AdminRequest;
use App\Models\Admin;
use App\Models\Country;
use App\Models\State;
use Illuminate\Contracts\Support\Renderable;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;
use Spatie\Permission\Models\Role;
use Illuminate\Http\Request;
use Yajra\DataTables\DataTables;
use Auth;

class AdminsController extends Controller
{
    public function index(): Renderable
	{
		// Ensure the user has the 'investor.view' permission
		$this->checkAuthorization(auth()->user(), ['investor.view']);

		// Get all channel partners with 'investor' role
		$investor = Admin::whereHas('roles', function ($q) {
			$q->where('name', 'investor');
		})->get();

		return view('backend.pages.admins.index', [
			'admins' => $investor,
			'visited' => []
		]);
	}

    public function create(): Renderable
    {
        $this->checkAuthorization(auth()->user(), ['investor.create']);
		$countries = Country::all();
		$roles = Role::all();
        return view('backend.pages.admins.create', compact('roles', 'countries'));
    }
	public function uploaddocument($id)
    {
       
		$admin = Admin::find($id);
        return view('backend.pages.admins.uploaddocuments', compact('admin'));
    }
	public function postuploadDocuments(Request $request)
{
    try {
        // Check if Admin Exists
        $admin = Admin::find($request->admin_id);
        if (!$admin) {
            return redirect()->back()->with('error', 'Admin not found!');
        }

        // Validate Files (Ensuring They Are Image Types & Max Size 2MB)
		$request->validate([
			'aadhar_card_doc' => 'nullable|file|mimes:pdf,jpg,jpeg,png,gif|max:2048',
			'pan_card_doc' => 'nullable|file|mimes:pdf,jpg,jpeg,png,gif|max:2048',
			'company_gst_certificate' => 'nullable|file|mimes:pdf,jpg,jpeg,png,gif|max:2048',
			'company_pan' => 'nullable|file|mimes:pdf,jpg,jpeg,png,gif|max:2048',
		]);


        // Set Upload Path
        $uploadPath = public_path('documents/');
        if (!file_exists($uploadPath)) {
            mkdir($uploadPath, 0775, true); // Ensure directory exists
        }

        // Fields for Upload
        $fields = [
            'aadhar_card_doc' => 'aadhar',
            'pan_card_doc' => 'pan',
            'company_gst_certificate' => 'gst',
            'company_pan' => 'company_pan',
        ];

        foreach ($fields as $fieldName => $prefix) {
            if ($request->hasFile($fieldName)) {
                $file = $request->file($fieldName);

                // Check if File is Valid
                if (!$file->isValid()) {
                    \Log::error("File upload failed for: $fieldName | Error: " . $file->getErrorMessage());
                    return redirect()->back()->with('error', "The $fieldName failed to upload. Error: " . $file->getErrorMessage());
                }

                // Generate Unique Filename
                $filename = uniqid() . '_' . time() . '_' . $prefix . '.' . $file->getClientOriginalExtension();
                
                // Move File to Upload Directory
                if ($file->move($uploadPath, $filename)) {
                    $admin->$fieldName = 'documents/' . $filename; // Save Path in DB
                } else {
                    \Log::error("Failed to move file: $fieldName");
                    return redirect()->back()->with('error', "The $fieldName could not be saved.");
                }
            }
        }

        // Save Admin Record
        $admin->save();
        return redirect()->back()->with('success', 'Documents uploaded successfully!');

    } catch (\Exception $e) {
        \Log::error("File Upload Exception: " . $e->getMessage());
        return redirect()->back()->with('error', $e->getMessage().'document must not be larger than 2MB.');
    }
}




    public function store(AdminRequest $request): RedirectResponse
    {
        $this->checkAuthorization(auth()->user(), ['investor.create']);
		$generateReferralId = $this->generateInvestorReferralId();
        $admin = new Admin();
        $admin->title = $request->title; 
        $admin->username = $request->username; 
        $admin->first_name = $request->first_name; 
        $admin->middle_name = $request->middle_name; 
        $admin->last_name = $request->last_name; 
        $admin->father_name = $request->father_name; 
        $admin->mobile = $request->mobile;
		$admin->whatsapp = $request->whatsapp; 
        $admin->email = $request->email;
        $admin->address = $request->address; 
        $admin->country_id = $request->country; 
        $admin->state_id = $request->state; 
        $admin->city = $request->city; 
        $admin->pincode = $request->pincode; 
        //$admin->years_of_exp = $request->years_of_exp; 
        $admin->company_name = $request->company_name; 
        $admin->company_gst = $request->company_gst; 
        $admin->education_qualification = $request->education_qualification; 
        $admin->my_referral_id = $request->my_referral_id; 
        $admin->referral_id = $generateReferralId; // Generate or assign your logic

        // Add nominee details
        $admin->nominee_name = $request->nominee_name;
        $admin->nominee_relationship = $request->nominee_relationship;
        $admin->nominee_contact = $request->nominee_contact;

        // Add bank details
        $admin->bank_name = $request->bank_name;
        $admin->account_number = $request->account_number;
        $admin->ifsc_code = $request->ifsc_code;
        $admin->account_type = $request->account_type;
		$admin->pan = $request->pan; 
		$admin->aadhar_number = $request->aadhar_number; // New field
        $admin->password = Hash::make($request->password);
        $admin->name = trim("{$request->first_name} {$request->middle_name} {$request->last_name}");
        $admin->save();

       $admin->assignRole(['investor']);
       

        session()->flash('success', __('Investor ('.$admin->referral_id.') has been created.'));
        return redirect()->route('admin.admins.index');
    }

    public function edit(int $id): Renderable
    {
        $this->checkAuthorization(auth()->user(), ['investor.edit']);

        $admin = Admin::findOrFail($id);
        $countries = Country::all();
        $roles = Role::all();
        $states = State::all();
        return view('backend.pages.admins.edit', compact('admin', 'countries', 'roles', 'states'));
    }

    public function update(AdminRequest $request, int $id): RedirectResponse
    {
        $this->checkAuthorization(auth()->user(), ['investor.edit']);

        $admin = Admin::findOrFail($id);
        $admin->username = $request->username; 
        $admin->title = $request->title; 
        $admin->first_name = $request->first_name; 
        $admin->middle_name = $request->middle_name; 
        $admin->last_name = $request->last_name; 
        $admin->father_name = $request->father_name; 
        $admin->mobile = $request->mobile;
        $admin->email = $request->email;
        $admin->whatsapp = $request->whatsapp; 
        $admin->address = $request->address; 
        $admin->country_id = $request->country; 
        $admin->state_id = $request->state; 
        $admin->city = $request->city; 
        $admin->pincode = $request->pincode; 
        //$admin->years_of_exp = $request->years_of_exp; 
        $admin->company_name = $request->company_name; 
        $admin->company_gst = $request->company_gst; 
        $admin->education_qualification = $request->education_qualification; 
        $admin->my_referral_id = $request->my_referral_id; 

        // Update nominee details
        $admin->nominee_name = $request->nominee_name;
        $admin->nominee_relationship = $request->nominee_relationship;
        $admin->nominee_contact = $request->nominee_contact;

        // Update bank details
        $admin->bank_name = $request->bank_name;
        $admin->account_number = $request->account_number;
        $admin->ifsc_code = $request->ifsc_code;
        $admin->account_type = $request->account_type;
		$admin->pan = $request->pan; 
		$admin->aadhar_number = $request->aadhar_number; // New field
        $admin->name = trim("{$request->first_name} {$request->middle_name} {$request->last_name}");
        if ($request->password) {
            $admin->password = Hash::make($request->password);
        }
        
        $admin->save();

        

        session()->flash('success', 'investor has been updated.');
        return back();
    }

    public function destroy(int $id): RedirectResponse
    {
        $this->checkAuthorization(auth()->user(), ['investor.delete']);

        $admin = Admin::findOrFail($id);
        $admin->delete();
        session()->flash('success', 'investor has been deleted.');
        return back();
    }

   protected function generateInvestorReferralId()
	{
		$prefix = 'ASSATZ';
		// Get the last referral ID from the Admin table
		$lastReferral = Admin::where('referral_id', 'LIKE', "{$prefix}%")->whereHas('roles',function($q){ $q->where('name','investor');})->orderBy('referral_id', 'desc')->first();

		// Extract the numeric part and increment it
		if (!empty($lastReferral->referral_id)) {
			// Get the numeric part after the prefix (e.g., "GIOCP" or "FR")
			$lastId = intval(substr($lastReferral->referral_id, strlen($prefix))); 
			// Increment the ID and pad with zeros to ensure 5 digits
			$newId = str_pad((string)($lastId + 1), 4, '0', STR_PAD_LEFT); 
		} else {
			// If no referral ID exists, start from 00001
			$newId = '1001';
		}

		// Return the new Referral ID with the prefix
		return $prefix . $newId;
	}


    public function getStates(Request $request)
    {
        $countryId = $request->input('country_id');
        $states = State::where('country_id', $countryId)->select('state_id', 'name')->get();
        return response()->json($states);
    }
	
	

    public function myteamindex(Request $request)
    {
        if ($request->ajax()) {
			 $usr = Auth::guard('admin')->user();
            // Fetch admin data for Datatables
            $admins = Admin::Query()->where('my_referral_id',$usr->referral_id);
            
            return DataTables::of($admins)
                
                //->rawColumns() // Make the action column raw HTML
                ->make(true); // Return the Datatables response in JSON format
        }

        // If it's not an AJAX request, return the normal view
        return view('backend.pages.myteam.index');
    }
	public function getAdminDetails(Request $request)
	{
		$referralId = $request->input('referral_id');

		// Find the admin using the referral ID
		$admin = Admin::where('referral_id', $referralId)->first();

		if ($admin) {
			return response()->json([
				'name' => $admin->name,
				'email' => $admin->email,
				// Add more fields if needed
			]);
		} else {
			return response()->json(['error' => 'No data found with this Referral ID'], 404);
		}
	}
	public function deleteDocument($adminId, $doc)
{
    $admin = Admin::find($adminId);

    if (!$admin) {
        return response()->json(['success' => false, 'message' => 'Admin not found'], 404);
    }

    
    $documentField = $doc;
    
    $admin->$documentField = null;
    $admin->save();

    return response()->json(['success' => true, 'message' => 'Document deleted successfully']);
}




}
