<?php

declare(strict_types=1);

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Http\Requests\UserRequest;
use App\Models\Project;
use Illuminate\Http\Request;
use Illuminate\Contracts\Support\Renderable;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\Hash;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Facades\Storage;

class ProjectController extends Controller
{
    public function index()
    {
        $projects = Project::latest()->get();
        return view('backend.pages.projects.index', compact('projects'));
    }

    public function create()
    {
        return view('backend.pages.projects.create');
    }

    public function store(Request $request)
	{
		$request->validate([
			'name' => 'required|string|max:255',
			'image' => 'required|image|mimes:jpg,png,jpeg,gif,svg|max:2048',
			'description' => 'nullable|string',
			'status' => 'required|in:Active,Inactive',
			'launching_soon' => 'nullable',
			'target_irr' => 'nullable|string',
			'asset_value' => 'nullable|string',
			'asset_type' => 'nullable|string',
			'minimum_investment' => 'nullable|string',
		]);

		$data = $request->all();
		$data['launching_soon'] = $request->has('launching_soon') ? 1 : 0;
        // Image upload using move() function
		if ($request->hasFile('image')) {
			$image = $request->file('image');
			$imageName = time() . '.' . $image->getClientOriginalExtension();
			$destinationPath = public_path('uploads/projects/');
			$image->move($destinationPath, $imageName);
			$data['image'] = 'uploads/projects/' . $imageName;
		}

        Project::create($data);

		return redirect()->route('admin.projects.index')->with('success', 'Project created successfully.');
	}

    public function show(Project $project)
    {
        return view('backend.pages.projects.show', compact('project'));
    }

    public function edit(Project $project)
    {
        return view('backend.pages.projects.edit', compact('project'));
    }

	public function update(Request $request, Project $project)
	{
		$request->validate([
			'name' => 'required|string|max:255',
			'image' => 'nullable|image|mimes:jpg,png,jpeg,gif,svg|max:2048',
			'description' => 'nullable|string',
			'status' => 'required|in:Active,Inactive',
			'launching_soon' => 'nullable',
			'target_irr' => 'nullable|string',
			'asset_value' => 'nullable|string',
			'asset_type' => 'nullable|string',
			'minimum_investment' => 'nullable|string',
		]);

		 $data = $request->all();
		$data['launching_soon'] = $request->has('launching_soon') ? 1 : 0;
        // Image upload using move() function
		if ($request->hasFile('image')) {
			// Delete old image if exists
			if ($project->image && file_exists(public_path($project->image))) {
				unlink(public_path($project->image));
			}

			$image = $request->file('image');
			$imageName = time() . '.' . $image->getClientOriginalExtension();
			$destinationPath = public_path('uploads/projects/');
			$image->move($destinationPath, $imageName);
			$data['image'] = 'uploads/projects/' . $imageName;
		}

        $project->update($data);


		return redirect()->route('admin.projects.index')->with('success', 'Project updated successfully.');
	}

    public function destroy(Project $project)
    {
        $project->delete();

        return redirect()->route('admin.projects.index')->with('success', 'Project deleted successfully.');
    }
}
