<?php

declare(strict_types=1);

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Http\Requests\AdminRequest;
use App\Models\InvestorRecord;
use App\Models\Admin;
use App\Models\Country;
use App\Models\State;
use Illuminate\Contracts\Support\Renderable;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;
use Spatie\Permission\Models\Role;
use Illuminate\Http\Request;
use Yajra\DataTables\DataTables;
use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;
use Auth;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Crypt;

class AdminsController extends Controller
{
	

	public function storeInvestorApi(Request $request): JsonResponse
	{ 
	
		try {
			
			
			 $validator = Validator::make($request->all(), [
				'email' => [
					'required',
					'email',
					'max:100',
					function ($attribute, $value, $fail) {
						$existsInvestor = \DB::table('investor_records')->where('email', $value)->exists();
						$existsAdmin = \DB::table('admins')->where('email', $value)->exists();
						if ($existsInvestor || $existsAdmin) {
							$fail('The '.$attribute.' has already been taken.');
						}
					}
				],
			]);

			if ($validator->fails()) {
				return response()->json([
					'status' => 'error',
					'message' => $validator->errors()->first('email')
				], 422);
			}
		// Generate referral ID
			$generateReferralId = $this->generateInvestorReferralId();

			$investor = new InvestorRecord();
			$investor->title = $request->title;
			$investor->username = $request->username;
			$investor->first_name = $request->first_name;
			$investor->middle_name = $request->middle_name;
			$investor->last_name = $request->last_name;
			$investor->father_name = $request->father_name;
			$investor->mobile = $request->mobile;
			$investor->whatsapp = $request->whatsapp;
			$investor->email = $request->email;
			$investor->address = $request->address;
			$investor->country_id = $request->country_id;
			$investor->state_id = $request->state_id;
			$investor->city = $request->city;
			$investor->pincode = $request->pincode;
			$investor->company_name = $request->company_name;
			$investor->company_gst = $request->company_gst;
			$investor->education_qualification = $request->education_qualification;
			//$investor->referral_id = $generateReferralId;

			// Nominee details
			$investor->nominee_name = $request->nominee_name;
			$investor->nominee_relationship = $request->nominee_relationship;
			$investor->nominee_contact = $request->nominee_contact;

			// Bank details
			$investor->bank_name = $request->bank_name;
			$investor->bank_branch = $request->bank_branch;
			$investor->account_number = $request->account_number;
			$investor->ifsc_code = $request->ifsc_code;
			$investor->account_type = $request->account_type;
			$investor->pan = $request->pan;
			$investor->aadhar_number = $request->aadhar_number;

			// Added by fields
			$investor->added_by_name = $request->added_by_name;
			$investor->added_by_email = $request->added_by_email;
			$investor->added_by_personal_id = $request->added_by_personal_id;

			// Status (default 'pending' if not provided)
			$investor->status = $request->status ?? 'Unverified';

			$investor->password = Hash::make($request->password ?? "123456");
			$investor->name = trim("{$request->first_name} {$request->middle_name} {$request->last_name}");

			$investor->save();
			
			
				// Mail send
				$encryptedId = Crypt::encryptString($investor->id);
			$approvalUrl = url("/investor/approve/{$encryptedId}");

			$subject = "Welcome to Assatz - Please Verify Your Account";

			$body = "
			<div style='font-family: Arial, sans-serif; background-color: #4b0561; padding:20px;'>
			  <div style='max-width:600px; margin:0 auto; background:#ffffff; border-radius:8px; overflow:hidden; box-shadow:0 2px 8px rgba(0,0,0,0.1);'>
				
				<!-- Header -->
				<div style='background:#4b0561; color:#ffffff; padding:20px; text-align:center;'>
				  <h2 style='margin:0;'>Welcome to Assatz</h2>
				</div>
				
				<!-- Body -->
				<div style='padding:25px; color:#333333; line-height:1.6;'>
				  <p>Dear <strong>{$investor->name}</strong>,</p>
				  
				  <p>We are excited to welcome you to <strong>Assatz</strong>! Your investor account has been successfully created, but it still needs to be verified.</p>
				  
				  <p>To activate your account and get started, please click the button below:</p>
				  
				  <p style='text-align:center; margin:30px 0;'>
					<a href='{$approvalUrl}' target='_blank' 
					   style='background:#4b0561; color:#ffffff; padding:14px 28px; text-decoration:none; border-radius:5px; font-weight:bold; display:inline-block;'>
					  Approve & Activate Account
					</a>
				  </p>
				  
				  <p>If the button above doesn’t work, copy and paste the link below into your browser:</p>
				  <p><a href='{$approvalUrl}' target='_blank' style='color:#004080;'>{$approvalUrl}</a></p>
				  
				  <br>
				  <p>Once your account is approved, you can log in to your dashboard and start exploring investment opportunities with us.</p>
				  
				  <p>We’re delighted to have you on board and look forward to a great journey together.</p>
				  
				  <p>Warm regards,<br><strong>Assatz Team</strong></p>
				</div>
			  </div>
			</div>
			";

			$this->sendMail($investor->email, $subject, $body);

			
			return response()->json([
				'status' => 'success',
				'message' => "Investor ({$investor->name}) has been created.",
				'data' => $investor
			], 201);

		} catch (\Exception $e) {
			return response()->json([
				'status' => 'error',
				'message' => $e->getMessage()
			], 500);
		}
	}
	
	

	public function approveaccountmail(Request $request): JsonResponse
	{
		try {
			// Validate request
			$validator = Validator::make($request->all(), [
				'id' => 'required|integer|exists:investor_records,id',
			]);

			if ($validator->fails()) {
				return response()->json([
					'status'  => 'error',
					'message' => $validator->errors()
				], 422);
			}

			// Fetch investor
			$investor = InvestorRecord::find($request->id);
			if (!$investor) {
				return response()->json([
					'status'  => 'error',
					'message' => 'Investor not found.'
				], 404);
			}

			// Encrypt ID & prepare approval URL
			$encryptedId = Crypt::encryptString($investor->id);
			$approvalUrl = url("/investor/approve/{$encryptedId}");

			// Subject
			$subject = "Assatz - Verification Link Resent";

			$body = "
			<div style='font-family: Arial, sans-serif; background-color: #4b0561; padding:20px;'>
			  <div style='max-width:600px; margin:0 auto; background:#ffffff; border-radius:8px; overflow:hidden; box-shadow:0 2px 8px rgba(0,0,0,0.1);'>
				
				<!-- Header -->
				<div style='background:#4b0561; color:#ffffff; padding:20px; text-align:center;'>
				  <h2 style='margin:0;'>Account Verification - Assatz</h2>
				</div>
				
				<!-- Body -->
				<div style='padding:25px; color:#333333; line-height:1.6;'>
				  <p>Dear <strong>{$investor->name}</strong>,</p>
				  
				  <p>You previously received an invitation to verify your <strong>Assatz</strong> investor account.  
				  It seems the verification has not been completed yet, so we are resending the link for your convenience.</p>
				  
				  <p>Please verify your account by clicking the button below:</p>
				  
				  <p style='text-align:center; margin:30px 0;'>
					<a href='{$approvalUrl}' target='_blank' 
					   style='background:#4b0561; color:#ffffff; padding:14px 28px; text-decoration:none; border-radius:5px; font-weight:bold; display:inline-block;'>
					  Verify & Activate Account
					</a>
				  </p>
				  
				  <p>If the button above doesn’t work, copy and paste the link below into your browser:</p>
				  <p><a href='{$approvalUrl}' target='_blank' style='color:#004080;'>{$approvalUrl}</a></p>
				  
				  <br>
				  <p>Once verified, you will be able to log in to your dashboard and explore investment opportunities with us.</p>
				  
				  <p>We appreciate your prompt attention to this and look forward to having you onboard.</p>
				  
				  <p>Warm regards,<br><strong>Assatz Team</strong></p>
				</div>
			  </div>
			</div>
			";


			// Send mail
			$this->sendMail($investor->email, $subject, $body);

			// Response
			return response()->json([
				'status'  => 'success',
				'message' => "Approval email has been sent successfully.",
				'data'    => $investor
			], 201);

		} catch (\Exception $e) {
			return response()->json([
				'status'  => 'error',
				'message' => $e->getMessage()
			], 500);
		}
	}

	private function sendMail($to, $subject, $body)
    {
        try {
            $host = "smtpout.secureserver.net";
            $uName = "noreply@assatz.co.in";
            $pass = "Deepak@123456";

            $mail = new PHPMailer(true);
            $mail->isSMTP();
            $mail->Host = $host;
            $mail->SMTPAuth = true;
            $mail->Username = $uName;
            $mail->Password = $pass;
            $mail->SMTPSecure = 'ssl';
            $mail->Port = 465;

            $mail->SMTPOptions = [
                'ssl' => [
                    'verify_peer' => false,
                    'verify_peer_name' => false,
                    'allow_self_signed' => true,
                ]
            ];

            $mail->setFrom('noreply@assatz.co.in', 'Assatz');
            $mail->addAddress($to);

            $mail->isHTML(true);
            $mail->Subject = $subject;
            $mail->Body    = $body;

            if (!$mail->send()) {
                return ['success' => false, 'message' => 'Mail not sent: ' . $mail->ErrorInfo];
            } else {
                return ['success' => true, 'message' => 'Mail sent successfully!'];
            }
        } catch (Exception $e) {
            return ['success' => false, 'message' => 'Mail error: ' . $e->getMessage()];
        }
    }

    public function index(): Renderable
	{
		// Ensure the user has the 'investor.view' permission
		$this->checkAuthorization(auth()->user(), ['investor.view']);

		// Get all channel partners with 'investor' role
		$investor = Admin::whereHas('roles', function ($q) {
			$q->where('name', 'investor');
		})->get();

		return view('backend.pages.admins.index', [
			'admins' => $investor,
			'visited' => []
		]);
	}

    public function create(): Renderable
    {
        $this->checkAuthorization(auth()->user(), ['investor.create']);
		$countries = Country::all();
		$roles = Role::all();
        return view('backend.pages.admins.create', compact('roles', 'countries'));
    }
	public function uploaddocument($id)
    {
       
		$admin = Admin::find($id);
        return view('backend.pages.admins.uploaddocuments', compact('admin'));
    }
	public function postuploadDocuments(Request $request)
{
    try {
        // Check if Admin Exists
        $admin = Admin::find($request->admin_id);
        if (!$admin) {
            return redirect()->back()->with('error', 'Admin not found!');
        }

        // Validate Files (Ensuring They Are Image Types & Max Size 2MB)
		$request->validate([
			'aadhar_card_doc' => 'nullable|file|mimes:pdf,jpg,jpeg,png,gif|max:2048',
			'pan_card_doc' => 'nullable|file|mimes:pdf,jpg,jpeg,png,gif|max:2048',
			'company_gst_certificate' => 'nullable|file|mimes:pdf,jpg,jpeg,png,gif|max:2048',
			'company_pan' => 'nullable|file|mimes:pdf,jpg,jpeg,png,gif|max:2048',
		]);


        // Set Upload Path
        $uploadPath = public_path('documents/');
        if (!file_exists($uploadPath)) {
            mkdir($uploadPath, 0775, true); // Ensure directory exists
        }

        // Fields for Upload
        $fields = [
            'aadhar_card_doc' => 'aadhar',
            'pan_card_doc' => 'pan',
            'company_gst_certificate' => 'gst',
            'company_pan' => 'company_pan',
        ];

        foreach ($fields as $fieldName => $prefix) {
            if ($request->hasFile($fieldName)) {
                $file = $request->file($fieldName);

                // Check if File is Valid
                if (!$file->isValid()) {
                    \Log::error("File upload failed for: $fieldName | Error: " . $file->getErrorMessage());
                    return redirect()->back()->with('error', "The $fieldName failed to upload. Error: " . $file->getErrorMessage());
                }

                // Generate Unique Filename
                $filename = uniqid() . '_' . time() . '_' . $prefix . '.' . $file->getClientOriginalExtension();
                
                // Move File to Upload Directory
                if ($file->move($uploadPath, $filename)) {
                    $admin->$fieldName = 'documents/' . $filename; // Save Path in DB
                } else {
                    \Log::error("Failed to move file: $fieldName");
                    return redirect()->back()->with('error', "The $fieldName could not be saved.");
                }
            }
        }

        // Save Admin Record
        $admin->save();
		
		$this->updateKycStatus($admin);
        return redirect()->back()->with('success', 'Documents uploaded successfully!');

    } catch (\Exception $e) {
        \Log::error("File Upload Exception: " . $e->getMessage());
        return redirect()->back()->with('error', $e->getMessage().'document must not be larger than 2MB.');
    }
}

	private function updateKycStatus($user)
	{
		$investorRecord = InvestorRecord::where('email', $user->email)->first();
		if (!$investorRecord) {
			return; // No record found, exit
		}
		
		$status = 'KYC pending';
		$hasAadhar = !empty($user->aadhar_card_doc);
		$hasPan = !empty($user->pan_card_doc);
		

		if ($hasAadhar && $hasPan ) {
			$status = 'KYC updated';
		} elseif ($hasAadhar || $hasPan ) {
			$status = 'KYC pending';
		}

		if ($investorRecord->kyc_status != $status) {
			$investorRecord->kyc_status = $status;
			$investorRecord->save();
		}
	}



    public function store(AdminRequest $request): RedirectResponse
    {
        $this->checkAuthorization(auth()->user(), ['investor.create']);
		$generateReferralId = $this->generateInvestorReferralId();
        $admin = new Admin();
        $admin->title = $request->title; 
        $admin->username = $request->username; 
        $admin->first_name = $request->first_name; 
        $admin->middle_name = $request->middle_name; 
        $admin->last_name = $request->last_name; 
        $admin->father_name = $request->father_name; 
        $admin->mobile = $request->mobile;
		$admin->whatsapp = $request->whatsapp; 
        $admin->email = $request->email;
        $admin->address = $request->address; 
        $admin->country_id = $request->country; 
        $admin->state_id = $request->state; 
        $admin->city = $request->city; 
        $admin->pincode = $request->pincode; 
        //$admin->years_of_exp = $request->years_of_exp; 
        $admin->company_name = $request->company_name; 
        $admin->company_gst = $request->company_gst; 
        $admin->education_qualification = $request->education_qualification; 
        $admin->my_referral_id = $request->my_referral_id; 
        $admin->referral_id = $generateReferralId; // Generate or assign your logic

        // Add nominee details
        $admin->nominee_name = $request->nominee_name;
        $admin->nominee_relationship = $request->nominee_relationship;
        $admin->nominee_contact = $request->nominee_contact;

        // Add bank details
        $admin->bank_name = $request->bank_name;
        $admin->account_number = $request->account_number;
        $admin->ifsc_code = $request->ifsc_code;
        $admin->account_type = $request->account_type;
		$admin->pan = $request->pan; 
		$admin->aadhar_number = $request->aadhar_number; // New field
        $admin->password = Hash::make($request->password);
        $admin->name = trim("{$request->first_name} {$request->middle_name} {$request->last_name}");
        $admin->save();

       $admin->assignRole(['investor']);
       

        session()->flash('success', __('Investor ('.$admin->referral_id.') has been created.'));
        return redirect()->route('admin.admins.index');
    }

    public function edit(int $id): Renderable
    {
        $this->checkAuthorization(auth()->user(), ['investor.edit']);

        $admin = Admin::findOrFail($id);
        $countries = Country::all();
        $roles = Role::all();
        $states = State::all();
        return view('backend.pages.admins.edit', compact('admin', 'countries', 'roles', 'states'));
    }

    public function update(AdminRequest $request, int $id): RedirectResponse
    {
        $this->checkAuthorization(auth()->user(), ['investor.edit']);

        $admin = Admin::findOrFail($id);
        $admin->username = $request->username; 
        $admin->title = $request->title; 
        $admin->first_name = $request->first_name; 
        $admin->middle_name = $request->middle_name; 
        $admin->last_name = $request->last_name; 
        $admin->father_name = $request->father_name; 
        $admin->mobile = $request->mobile;
        $admin->email = $request->email;
        $admin->whatsapp = $request->whatsapp; 
        $admin->address = $request->address; 
        $admin->country_id = $request->country; 
        $admin->state_id = $request->state; 
        $admin->city = $request->city; 
        $admin->pincode = $request->pincode; 
        //$admin->years_of_exp = $request->years_of_exp; 
        $admin->company_name = $request->company_name; 
        $admin->company_gst = $request->company_gst; 
        $admin->education_qualification = $request->education_qualification; 
        $admin->my_referral_id = $request->my_referral_id; 

        // Update nominee details
        $admin->nominee_name = $request->nominee_name;
        $admin->nominee_relationship = $request->nominee_relationship;
        $admin->nominee_contact = $request->nominee_contact;

        // Update bank details
        $admin->bank_name = $request->bank_name;
        $admin->account_number = $request->account_number;
        $admin->ifsc_code = $request->ifsc_code;
        $admin->account_type = $request->account_type;
		$admin->pan = $request->pan; 
		$admin->aadhar_number = $request->aadhar_number; // New field
        $admin->name = trim("{$request->first_name} {$request->middle_name} {$request->last_name}");
        if ($request->password) {
            $admin->password = Hash::make($request->password);
        }
        
        $admin->save();

        

        session()->flash('success', 'investor has been updated.');
        return back();
    }

    public function destroy(int $id): RedirectResponse
    {
        $this->checkAuthorization(auth()->user(), ['investor.delete']);

        $admin = Admin::findOrFail($id);
        $admin->delete();
        session()->flash('success', 'investor has been deleted.');
        return back();
    }

   protected function generateInvestorReferralId()
	{
		$prefix = 'ASSATZ';
		// Get the last referral ID from the Admin table
		$lastReferral = Admin::where('referral_id', 'LIKE', "{$prefix}%")->whereHas('roles',function($q){ $q->where('name','investor');})->orderBy('referral_id', 'desc')->first();

		// Extract the numeric part and increment it
		if (!empty($lastReferral->referral_id)) {
			// Get the numeric part after the prefix (e.g., "GIOCP" or "FR")
			$lastId = intval(substr($lastReferral->referral_id, strlen($prefix))); 
			// Increment the ID and pad with zeros to ensure 5 digits
			$newId = str_pad((string)($lastId + 1), 4, '0', STR_PAD_LEFT); 
		} else {
			// If no referral ID exists, start from 00001
			$newId = '1001';
		}

		// Return the new Referral ID with the prefix
		return $prefix . $newId;
	}


    public function getStates(Request $request)
    {
        $countryId = $request->input('country_id');
        $states = State::where('country_id', $countryId)->select('state_id', 'name')->get();
        return response()->json($states);
    }
	
	

    public function myteamindex(Request $request)
    {
        if ($request->ajax()) {
			 $usr = Auth::guard('admin')->user();
            // Fetch admin data for Datatables
            $admins = Admin::Query()->where('my_referral_id',$usr->referral_id);
            
            return DataTables::of($admins)
                
                //->rawColumns() // Make the action column raw HTML
                ->make(true); // Return the Datatables response in JSON format
        }

        // If it's not an AJAX request, return the normal view
        return view('backend.pages.myteam.index');
    }
	public function getAdminDetails(Request $request)
	{
		$referralId = $request->input('referral_id');

		// Find the admin using the referral ID
		$admin = Admin::where('referral_id', $referralId)->first();

		if ($admin) {
			return response()->json([
				'name' => $admin->name,
				'email' => $admin->email,
				// Add more fields if needed
			]);
		} else {
			return response()->json(['error' => 'No data found with this Referral ID'], 404);
		}
	}
	public function deleteDocument($adminId, $doc)
{
    $admin = Admin::find($adminId);

    if (!$admin) {
        return response()->json(['success' => false, 'message' => 'Admin not found'], 404);
    }

    
    $documentField = $doc;
    
    $admin->$documentField = null;
    $admin->save();

    return response()->json(['success' => true, 'message' => 'Document deleted successfully']);
}




}
