<?php

declare(strict_types=1);

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Models\Project;
use App\Models\ProjectVideo;
use App\Models\ProjectProgress;
use App\Models\ProjectProgressImage;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;

class ProjectController extends Controller
{
    public function index()
    {
        $projects = Project::latest()->get();
        return view('backend.pages.projects.index', compact('projects'));
    }

    public function create()
    {
        return view('backend.pages.projects.create');
    }
 
   public function store(Request $request)
{
    $request->validate([
        'name' => 'required|string|max:255',
        'image' => 'required|image|mimes:jpg,png,jpeg,gif,svg|max:2048',
        'description' => 'nullable|string',
        'status' => 'required|in:Active,Inactive',
        'launching_soon' => 'nullable',
        'target_irr' => 'nullable|string',
        'asset_value' => 'nullable|string',
        'asset_type' => 'nullable|string',
		 'balance_sheet' => 'file|mimes:pdf',
            // 'term_condition' => 'required|file|mimes:pdf',
            'term_condition' => 'required|file|mimes:pdf',
        'minimum_investment' => 'nullable|string',
        'progress_dates' => 'nullable|array',
        'progress_percentages' => 'nullable|array',
        'progress_images.*' => 'nullable|array',
        'progress_images.*.*' => 'image|mimes:jpg,png,jpeg,gif,svg|max:2048',
        //'progress_titles.*.*' => 'nullable|string|max:255',
		  // Video Section
        'video_titles' => 'nullable|array',
        'video_titles.*' => 'nullable|string|max:255',
        'video_links' => 'nullable|array',
        'video_links.*' => 'nullable|url',
    ]);

    $data = $request->all();
    //$data['launching_soon'] = $request->has('launching_soon') ? 1 : 0;

    if ($request->hasFile('image')) {
        $image = $request->file('image');
        $imageName = time() . '.' . $image->getClientOriginalExtension();
        $image->move(public_path('uploads/projects/'), $imageName);
        $data['image'] = 'uploads/projects/' . $imageName;
    }
	
       

		if ($request->hasFile('term_condition')) {
			$file = $request->file('term_condition');
			$filename = time() . '_terms.' . $file->getClientOriginalExtension();
			$file->move(public_path('terms'), $filename);
			$data['term_condition'] = 'terms/' . $filename;
		}
		if ($request->hasFile('balance_sheet')) {
			$file = $request->file('balance_sheet');
			$filename = time() . '_balancesheet.' . $file->getClientOriginalExtension();
			$file->move(public_path('balancesheet'), $filename);
			$data['balance_sheet'] = 'balancesheet/' . $filename;
		}

    $project = Project::create($data);

    if ($request->has('progress_dates')) {
        foreach ($request->progress_dates as $key => $date) {
            if (!isset($request->progress_percentages[$key])) {
                continue;
            }

            $progress = ProjectProgress::create([
                'project_id' => $project->id,
                'date' => $date,
                'progress_percentage' => $request->progress_percentages[$key] ?? null,
            ]);

            if ($request->hasFile("progress_images.$key")) {
                foreach ($request->file("progress_images.$key") as $index => $file) {
                    $filename = time() . '_' . $file->getClientOriginalName();
                    $file->move(public_path('progress_images'), $filename);
                    $imagePath = 'progress_images/' . $filename;
//print_r($request->progress_titles[$key][$index][$index]); exit;
                    ProjectProgressImage::create([
                        'progress_id' => $progress->id,
                        'image' => $imagePath,
                        'title' => $request->progress_titles[$key][$index][$index] ?? null, // Save image title
                    ]);
                }
            }
        }
    }
	
	   // ✅ Save project videos (NEW!)
    if ($request->has('video_titles') && $request->has('video_links')) {
        foreach ($request->video_titles as $i => $videoTitle) {
            $videoLink = $request->video_links[$i] ?? null;
            if ($videoTitle && $videoLink) {
                \App\Models\ProjectVideo::create([
                    'project_id' => $project->id,
                    'title' => $videoTitle,
                    'link' => $videoLink,
                ]);
            }
        }
    }

    return redirect()->route('admin.projects.index')->with('success', 'Project created successfully.');
}



    public function show(Project $project)
    {
        return view('backend.pages.projects.show', compact('project'));
    }

    public function edit(Project $project)
    {
        return view('backend.pages.projects.edit', compact('project'));
    }

 



	public function update(Request $request, Project $project)
	{
		$request->validate([
			'name' => 'required|string|max:255',
			'image' => 'nullable|image|mimes:jpg,png,jpeg,gif,svg|max:2048',
			'description' => 'nullable|string',
			'status' => 'required|in:Active,Inactive',
			 'balance_sheet' => 'nullable|file|mimes:pdf',
            'term_condition' => 'nullable|file|mimes:pdf',
			'launching_soon' => 'nullable',
			'target_irr' => 'nullable|string',
			'asset_value' => 'nullable|string',
			'asset_type' => 'nullable|string',
			'minimum_investment' => 'nullable|string',
			'progress_dates' => 'nullable|array',
			'progress_percentages' => 'nullable|array',
			'progress_images.*' => 'nullable|array',
			'progress_images.*.*' => 'image|mimes:jpg,png,jpeg,gif,svg|max:2048',
			'progress_titles.*' => 'nullable|array',
			'progress_titles.*.*' => 'nullable|string|max:255',
			 'video_titles.*' => 'nullable|string|max:255',
        'video_links.*' => 'nullable|url',
		]);

		$data = $request->only([
			'name', 'description', 'status', 'target_irr', 'roi', 'asset_value', 'launching_soon',
			'asset_type', 'minimum_investment'
		]);
		//$data['launching_soon'] = $request->has('launching_soon') ? 1 : 0;

		// Handle Main Image Update
		if ($request->hasFile('image')) {
			if (!empty($project->image) && File::exists(public_path($project->image))) {
				//File::delete(public_path($project->image));
			}
			$image = $request->file('image');
			$imageName = time() . '.' . $image->getClientOriginalExtension();
			$image->move(public_path('uploads/projects/'), $imageName);
			$data['image'] = 'uploads/projects/' . $imageName;
		}
		if ($request->hasFile('balance_sheet')) {
			$file = $request->file('balance_sheet');
			$filename = time() . '_balancesheet.' . $file->getClientOriginalExtension();
			$file->move(public_path('balancesheet'), $filename);
			$data['balance_sheet'] = 'balancesheet/' . $filename;
		}

		if ($request->hasFile('term_condition')) {
			$file = $request->file('term_condition');
			$filename = time() . '_terms.' . $file->getClientOriginalExtension();
			$file->move(public_path('terms'), $filename);
			$data['term_condition'] = 'terms/' . $filename;
		}


		$project->update($data);

		// Handle Project Progress Updates
		if ($request->has('progress_dates')) {
			foreach ($request->progress_dates as $key => $date) {
				$progress = isset($request->progress_ids[$key]) ? ProjectProgress::find($request->progress_ids[$key]) : new ProjectProgress();
				if (!$progress) {
					continue;
				}

				$progress->project_id = $project->id;
				$progress->date = $date;
				$progress->progress_percentage = $request->progress_percentages[$key] ?? 0;
				$progress->save();

				// Update Existing Image Titles
				if (!empty($request->progress_titles[$key])) {
					foreach ($request->progress_titles[$key] as $index => $title) {
						$existingImage = ProjectProgressImage::where('progress_id', $progress->id)->skip($index)->first();
						if ($existingImage) {
							$existingImage->update(['title' => $title ?? '']); // Ensure it's a string
						}
					}
				}

				// If New Images Are Uploaded, Add Them
				if ($request->hasFile("progress_images.$key")) {
					foreach ($request->file("progress_images.$key") as $index => $file) {
						$filename = time() . '_' . $file->getClientOriginalName();
						$file->move(public_path('progress_images'), $filename);
						$imagePath = 'progress_images/' . $filename;

						ProjectProgressImage::create([
							'progress_id' => $progress->id,
							'image' => $imagePath,
							'title' => isset($request->progress_titles[$key][$index]) ? (string) $request->progress_titles[$key][$index] : '', 
						]);
					}
				}
			}
		}
		
		  // Handle Progress Video Updates
			if ($request->has('video_titles')) {
				foreach ($request->video_titles as $index => $title) {
					$videoId = isset($request->video_ids[$index]) ? $request->video_ids[$index] : null;
					$videoLink = $request->video_links[$index] ?? null;

					// If a video ID exists, update it; otherwise, create a new one
					$video = $videoId ? ProjectVideo::find($videoId) : new ProjectVideo();
					if ($video) {
						$video->project_id = $project->id;
						$video->title = $title;
						$video->link = $videoLink;
						$video->save();
					}
				}
			}

		return redirect()->route('admin.projects.index')->with('success', 'Project updated successfully.');
	}


	public function destroy(Project $project)
	{
		// Delete project image
		if ($project->image && File::exists(public_path($project->image))) {
			//File::delete(public_path($project->image));
		}
		// Delete project progress images
		foreach ($project->progressImages as $progress) {
			foreach ($progress->images as $image) {
				if ($image->image && File::exists(public_path($image->image))) {
					//File::delete(public_path($image->image));
				}
				$image->delete();
			}
			$progress->delete();
		}
		// Delete the project itself
		$project->delete();

		return redirect()->route('admin.projects.index')->with('success', 'Project deleted successfully.');
	}   

	public function removeProgressimage($id)
	{
		$project= ProjectProgressImage::find($id);
		if ($project->image && File::exists(public_path($project->image))) {
			//File::delete(public_path($project->image));
		}
		$project->delete();
		return redirect()->back()->with('success', 'image deleted successfully.');
	}

	public function deleteProgress(ProjectProgress $progress)
	{
		// Delete all images related to this progress
		foreach ($progress->images as $image) {
			if ($image->image && File::exists(public_path($image->image))) {
				//File::delete(public_path($image->image));
			}
			$image->delete();
		}

		// Delete the progress record
		$progress->delete();

		return response()->json(['success' => true], 200);
	}


}
