import React, { useEffect, useState } from "react";
import { Row, Col, Alert } from "reactstrap";
import axios from "axios";
import Link from "next/link";
import "./projects.css";  

const FinancialProjectTab = () => {
  const BASE_URL = process.env.NEXT_PUBLIC_BASE_URL;
  const [investments, setInvestments] = useState([]);  
  const [successMessage, setSuccessMessage] = useState("");
  const [errorMessage, setErrorMessage] = useState("");

  useEffect(() => {
    const fetchInvestments = async () => {
      const token = localStorage.getItem("token");
      if (!token) {
        setErrorMessage("You need to be logged in to view the investments.");
        return;
      }

      try {
        const response = await axios.post(
          `${BASE_URL}/investments`, 
          {},  
          { headers: { Authorization: `Bearer ${token}` } }
        );

        setInvestments(response.data?.data || []);
      } catch (error) {
        console.error("Error fetching investments:", error);
        setErrorMessage(
          "Error fetching investments: " +
            (error.response?.data?.message || "Unknown error occurred.")
        );
      }
    };

    fetchInvestments();
  }, []);

  return (
    <div className="dashboard-content">
      <div className="projects-title">
        <h2>Financials</h2>
      </div>

      <div className="my-projects">
        {successMessage && <Alert color="success">{successMessage}</Alert>}
        {errorMessage && <Alert color="danger">{errorMessage}</Alert>}

        <div className="projects-listing">
          <Row>
            {investments.length > 0 ? (
              investments.map((investment) => (
                <Col md="6" key={investment.id} className="mb-4">
                  <div className="project-card">
                    {/* Project Image */}
                    <div className="investment-image mb-3">
                      <img 
                        src={investment.project.image_url || "/default-image.jpg"} 
                        alt={investment.project.name || "Unnamed Project"} 
                        className="img-fluid" 
                      />
					  <h5>{investment.project.name || "Unnamed Project"}</h5>
					<p><strong>Amount Invested:</strong> ${investment.amount_invested}</p>
					  <p><strong>Share Price:</strong> ${investment.share_price}</p>
					  <p><strong>Transaction Number:</strong> {investment.transaction_number}</p>
					  <p><strong>Transaction Date:</strong> {investment.transaction_date}</p>
                    </div>

                    

                    {/* New fields: Share Price and Share Certificate */}
                    <div className="investment-share-info">
                      
					  
                 
                      {investment.share_certificate && (
                        <Link  target="_blank"  href={investment.share_certificate_url} passHref>
                          <button className="btn btn-success  btn-sm">View Share Certificate</button>
                        </Link>
                      )} 
					  {investment.term_condition_url && (
                        <Link  target="_blank"  href={investment.term_condition_url} passHref>
                          <button className="btn btn-success btn-sm ml-2">View Terms & Conditions</button>
                        </Link>
                      )}
                    </div>

                    
                  </div>
                </Col>
              ))
            ) : (
              <Col md="12">
                <p>No investments found.</p>
              </Col>
            )}
          </Row>
        </div>
      </div>
    </div>
  );
};

export default FinancialProjectTab;
