import React, { useEffect, useState } from "react";
import { Row, Col, Alert } from "reactstrap";
import axios from 'axios';
import Link from 'next/link';  // Corrected import for Next.js Link
import './projects.css';  // Import global CSS
const AllProjectTab = () => {
  const BASE_URL = process.env.NEXT_PUBLIC_BASE_URL;
  const [getProjects, setProjects] = useState([]);  // Fixed state variable name
  const [successMessage, setSuccessMessage] = useState("");
  const [errorMessage, setErrorMessage] = useState("");

  // Fetch projects using POST method
  useEffect(() => {
    const fetchProjects = async () => {
      const token = localStorage.getItem('token');
      if (!token) {
        setErrorMessage("You need to be logged in to view the projects.");
        return;
      }

      try {
        const response = await axios.post(
          `${BASE_URL}/projects`,
          {},  // Empty object or any params if required by the API
          {
            headers: {
              'Authorization': `Bearer ${token}`,
            },
          }
        );
		
        setProjects(response.data.data);  // Set the fetched projects
      } catch (error) {
        console.error("Error fetching projects:", error);
        setErrorMessage("Error fetching projects: " + (error.response?.data?.message || "Unknown error occurred."));
      }
    };

    fetchProjects();
  }, []);

  return (
    <div className="dashboard-content">
      {/* Title for All Projects */}
      <div className="projects-title">
        <h2>All Projects</h2>
      </div>

      <div className="my-projects" id="getprojects">
        {successMessage && <Alert color="success">{successMessage}</Alert>}
        {errorMessage && <Alert color="danger">{errorMessage}</Alert>}

        <div className="projects-listing">
          <Row>
            {getProjects.length > 0 ? (
              getProjects.map(project => (
                <Col md="4" key={project.id} className="mb-4">
                  <div className="project-card">
                    {/* Display Project Image */}
                    <div className="project-image mb-3">
                      <img src={project.image_url} alt={project.name} className="img-fluid" />
                    </div>

                    <h5 className="project-title">{project.name}</h5>
                    
                    {/* Display the new fields */}
                    <div className="project-launching-soon">
                      <strong>Launching Soon: </strong>{project.launching_soon ? "Yes" : "No"}
                    </div>
                    <div className="project-target-irr">
                      <strong>Target IRR: </strong>{project.target_irr}%
                    </div>
                    <div className="project-asset-value">
                      <strong>Asset Value: </strong>${project.asset_value}
                    </div>
                    <div className="project-asset-type">
                      <strong>Asset Type: </strong>{project.asset_type}
                    </div>
                    <div className="project-minimum-investment">
                      <strong>Minimum Investment: </strong>${project.minimum_investment}
                    </div>

                    {/* Link to the project details page */}
                   <Link href={`/pages/other-pages/projects/${project.id}`}>
                      <button className="btn btn-success">
                        View Details
                      </button>
                    </Link>
                  </div>
                </Col>
              ))
            ) : (
              <Col md="12">
                <p>No projects found.</p>
              </Col>
            )}
          </Row>
        </div>
      </div>
    </div>
  );
};

export default AllProjectTab;
