import React, { useEffect, useState } from "react";
import axios from "axios";
import { Table, Alert, Button } from "reactstrap";

const BASE_URL = process.env.NEXT_PUBLIC_BASE_URL || "";

const AllfrenchisepartnerTab = () => {
  const [teamData, setTeamData] = useState([]);
  const [errorMessage, setErrorMessage] = useState("");
  const [currentPage, setCurrentPage] = useState(1); // Pagination state
  const [pageSize, setPageSize] = useState(10); // Number of items per page

  const fetchTeamData = async (searchTerm = "", page = 1, pageSize = 10) => {
    const token = localStorage.getItem("token");

    if (!token) {
      setErrorMessage("You need to be logged in to view team data.");
      return;
    }

    try {
      const response = await axios.post(
        `${BASE_URL}/allusers`,
        {
          search: searchTerm, // Pass search term as part of request body
          role: "franchise", // Pass role as 'franchise'
          page, // Add pagination parameters
          pageSize, // Add pagination parameters
        },
        {
          headers: { Authorization: `Bearer ${token}` },
        }
      );

      if (response.data.message === "Unauthenticated.") {
        localStorage.clear();
        window.location.href = "/";
      }

      if (response.data.success) {
        setTeamData(response.data.data);
      } else {
        setErrorMessage("Failed to fetch team data.");
      }
    } catch (error) {
      console.error("Error fetching team data:", error);
      setErrorMessage("Error fetching team data.");

      if (error.response && error.response.data && error.response.data.message === 'Unauthenticated.') {
        localStorage.clear();
        window.location.href = '/';
      }
    }
  };

  const handleSearchChange = (event) => {
    const searchTerm = event.target.value;
    fetchTeamData(searchTerm, currentPage, pageSize); // Fetch data with the new search term
  };

  const handlePageChange = (pageNumber) => {
    setCurrentPage(pageNumber);
    fetchTeamData("", pageNumber, pageSize); // Fetch data for the new page
  };

  useEffect(() => {
    fetchTeamData("", currentPage, pageSize);
  }, [currentPage, pageSize]);

  return (
    <div className="team-data-table" id="my-team">
      {errorMessage && <Alert color="danger">{errorMessage}</Alert>}

      <div style={{ display: 'flex', justifyContent: 'center', alignItems: 'center' }}>
        <input
          type="text"
          placeholder="Search team"
          onChange={handleSearchChange}
          style={{
            padding: '8px 16px',
            fontSize: '16px',
            borderRadius: '4px',
            border: '1px solid #ccc',
          }}
        />
      </div>

      <Table striped responsive>
        <thead>
          <tr>
            <th>Franchise Partner Code</th>
            <th>First Name</th>
            <th>Last Name</th>
            <th>Mobile</th>
            <th>Email</th>
            <th>Address</th>
          </tr>
        </thead>
        <tbody>
          {teamData.length > 0 ? (
            teamData.map((member) => (
              <React.Fragment key={member.id}>
                <tr>
                  <td>{member.referral_id || "N/A"}</td>
                  <td>{member.first_name || "N/A"}</td>
                  <td>{member.last_name || "N/A"}</td>
                  <td>{member.mobile || "N/A"}</td>
                  <td>{member.email || "N/A"}</td>
                  <td>
                    {member.city || "N/A"}, {member.state?.name || "N/A"}
                  </td>
                </tr>
              </React.Fragment>
            ))
          ) : (
            <tr>
              <td colSpan="8" style={{ textAlign: "center" }}>
                Data Not Found
              </td>
            </tr>
          )}
        </tbody>
      </Table>

      {/* Pagination Controls */}
      <div style={{ display: "flex", justifyContent: "center", marginTop: "20px" }}>
        <Button
          onClick={() => handlePageChange(currentPage - 1)}
          disabled={currentPage === 1}
          style={{ marginRight: "10px" }}
        >
          Previous
        </Button>
        <Button
          onClick={() => handlePageChange(currentPage + 1)}
          disabled={teamData.length < pageSize}
        >
          Next
        </Button>
      </div>
    </div>
  );
};

export default AllfrenchisepartnerTab;
